function [u,ll] = laplacedirfdsor( xso, xne, u0, alpha, tolerance, ...
                                maxiter, ncheck)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                         %
%  [u,ll] = laplacedirfdsor( xso, xne, u0, alpha, tolerance, ...          %
%                                maxiter, ncheck)                         %
%                                                                         %
%  Resuelve la ecuacion de laplace en un dominio rectangular mediante     %
%  diferencias finitas. Las ecuaciones de las diferencias finitas se      %
%  resuelven por sobre-relajacion sucesiva (SOR). Supone condiciones      %
%  de contorno de Dirichlet y malla equiespaciada.                        %
%                                                                         %
%  ENTRADA                                                                %
%     xso   coordenadas del punto inferior izquierdo (suroeste). Array    %
%           de dos valores con las coordenadas xy del punto.              %
%     xne   idem para el punto superior derecho.                          %
%      nx   numero de puntos de la malla en las direcciones XY. Array     %
%           de dos valores.                                               %
%      u0   estimacion inicial de la solucion. Incluye tambien los        %
%           valores en los contornos, que no van a ser alterados.         %
%tolerance  tolerancia requerida (variacion relativa entre iteraciones    %
%           sucesivas                                                     %
%  maxiter  maximo numero de iteraciones permitido                        %
%  ncheck   numero de iteraciones tras las cuales se desea un 'informe'   %
%                                                                         %
%  SALIDA                                                                 %
%       u  solucion encontrada. Array de nx filas y ny columnas.          %
%      ll  numero de iteraciones realizadas                               %
%                                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

u  = u0;
nx = size(u);
uu = zeros(nx);
dx = xne - xso;
dx = dx./(nx-1);
epsilon =  0.5 / (dx(1)^2+dx(2)^2) * [ dx(2)^2 dx(1)^2] ;

done = false; % parametro de control del bucle
ll = 0;       % contador de iteraciones

kk = 0;       % contador auxiliara para visualizacion
epskk = ones(1,maxiter/ncheck)*tolerance;

figure(9); clf;
while ~done
    
    done = true;
    ll = ll + 1;
    if( ll > maxiter ), 
        disp('Demasiadas iteraciones');
        u = 0.0;
        break;
    end;
    
     uu = u;
    % Bucle en u. . (Gauss-Seidel)
    for ii = 2:nx(1)-1
        for jj=2:nx(2)-1
            u(ii,jj) = epsilon(1)*( u(ii+1, jj) + u(ii-1,jj) ) + ...
                       epsilon(2)*( u(ii, jj+1) + u(ii, jj-1) );
        end
    end
    % SOR
    u = ( 1.0 + alpha)*u - alpha*uu;
    
    % calculamos la maxima diferencia entre una iteracion y la
    % siguiente y la almacenamos en epsjj
    
    epsll =  sum(abs(uu-u)) / sum(abs(uu));
    if ( epsll > tolerance )
        done = false;
    end
    
    
    % cada ncheck iteraciones guardamos el error relativo y dibujamos
    % la solucion (la funcion rem devuelve el resto de la division).
    if( rem( ll-1 , ncheck ) == 0)
        %plot(x,y,'-','Color',[ (jj-1)/maxiter 0 0 ]);
        surf(u);
        view(80, 20);
        legend(sprintf('iteracion %d\n eps %e',ll,epsll));
        pause(0.2);
        kk = kk+1;
        epskk(kk) = epsll;
    end
end
fprintf('\n iteraciones = %d\n',ll);

figure(10); clf;
semilogy( 1:ncheck:(kk*ncheck), epskk(1:kk),'o');
xlabel('numero de iteracion'); ylabel('error relativo');

% % comparamos con la solucion analitica
% ff = matlabFunction( dsolve( 'D2y - 5*Dy + 10*y = 10*x','y(0)=0', ...
%         'y(1)=100','x') );
% errrel = max( abs( (y-ff(x))/y ));
% fprintf('\n error relativo del resultado %e\n', errrel);     
% fprintf('\n tolerancia %e\n', tolerance);
% 
% % dibujamos la solucion aproximada por Jacobi y la analitica
% figure(3);
% plot(x,y,'ro',x,ff(x),'-b');
% xlabel('x'); ylabel('y');
% legend('Solucion por Jacobi','Solucion analitica','Location','North');

